<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/functions.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Ensure we're sending JSON response
header('Content-Type: application/json');

if (!isset($_GET['id'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Product ID is required']);
    exit;
}

$product_id = intval($_GET['id']);

try {
    // Log the incoming request
    error_log("Fetching product with ID: " . $product_id);

    // Get product details
    $stmt = $pdo->prepare("
        SELECT 
            p.id,
            p.name,
            p.description,
            p.category_id,
            p.price,
            p.cost_price,
            p.quantity,
            p.alert_quantity,
            c.name as category_name 
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE p.id = ?
    ");
    
    $stmt->execute([$product_id]);
    $product = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$product) {
        error_log("Product not found with ID: " . $product_id);
        http_response_code(404);
        echo json_encode(['error' => 'Product not found']);
        exit;
    }

    // Ensure all numeric values are properly formatted
    $product['id'] = intval($product['id']);
    $product['category_id'] = intval($product['category_id']);
    $product['price'] = floatval($product['price']);
    $product['cost_price'] = floatval($product['cost_price']);
    $product['quantity'] = intval($product['quantity']);
    $product['alert_quantity'] = intval($product['alert_quantity']);

    // Log successful response
    error_log("Successfully fetched product: " . json_encode($product));
    
    // Send the response
    echo json_encode($product);
    exit;
    
} catch (Exception $e) {
    error_log("Error in get_product.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'error' => 'Failed to fetch product details',
        'message' => $e->getMessage()
    ]);
    exit;
} 